<?php

require_once __DIR__ . '/DB.php';

class Analytics {
    
    public static function track($postId = null) {
        // Basic bot detection
        if (self::isBot()) return;

        $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
        $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        // Simple Browser/Platform detection
        $browser = self::getBrowser($ua);
        $platform = self::getPlatform($ua);
        
        // GeoIP Lookup
        $country = null;
        $city = null;
        $region = null;

        // Only lookup if not localhost
        if ($ip !== '127.0.0.1' && $ip !== '::1') {
            $geoData = self::getGeoLocation($ip);
            if ($geoData) {
                $country = $geoData['countryCode'] ?? null;
                $city = $geoData['city'] ?? null;
                $region = $geoData['regionName'] ?? null;
            }
        }

        try {
            DB::insert('analytics_visits', [
                'post_id' => $postId,
                'ip_hash' => md5($ip), // Privacy friendly
                'user_agent' => substr($ua, 0, 255),
                'browser' => $browser,
                'platform' => $platform,
                'country_code' => $country,
                'city' => $city,
                'region' => $region
            ]);
            
            // Also update aggregate stats
            $today = date('Y-m-d');
            DB::query("INSERT INTO stats_daily (day, views) VALUES (?, 1) ON DUPLICATE KEY UPDATE views = views + 1", [$today]);
            
        } catch (Exception $e) {
            // Fail silently to not break the page
        }
    }

    private static function getGeoLocation($ip) {
        // Use ip-api.com (free for non-commercial use, 45 requests/min)
        $url = "http://ip-api.com/json/$ip?fields=status,countryCode,regionName,city";
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_TIMEOUT, 2); // Fast timeout
        $json = curl_exec($ch);
        curl_close($ch);
        
        if ($json) {
            $data = json_decode($json, true);
            if (isset($data['status']) && $data['status'] === 'success') {
                return $data;
            }
        }
        return null;
    }

    private static function isBot() {
        $ua = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
        $bots = ['bot', 'crawl', 'slurp', 'spider', 'mediapartners'];
        foreach ($bots as $bot) {
            if (strpos($ua, $bot) !== false) return true;
        }
        return false;
    }

    private static function getBrowser($ua) {
        $ua = strtolower($ua);
        if (strpos($ua, 'chrome') !== false) return 'Chrome';
        if (strpos($ua, 'firefox') !== false) return 'Firefox';
        if (strpos($ua, 'safari') !== false) return 'Safari';
        if (strpos($ua, 'edge') !== false) return 'Edge';
        if (strpos($ua, 'opera') !== false) return 'Opera';
        if (strpos($ua, 'msie') !== false || strpos($ua, 'trident') !== false) return 'IE';
        return 'Other';
    }

    private static function getPlatform($ua) {
        $ua = strtolower($ua);
        if (strpos($ua, 'windows') !== false) return 'Windows';
        if (strpos($ua, 'mac') !== false) return 'MacOS';
        if (strpos($ua, 'linux') !== false) return 'Linux';
        if (strpos($ua, 'android') !== false) return 'Android';
        if (strpos($ua, 'iphone') !== false || strpos($ua, 'ipad') !== false) return 'iOS';
        return 'Other';
    }
    
    public static function getStats($period = '24h') {
        $sql = "";
        $params = [];
        
        switch($period) {
            case '24h':
                $sql = "SELECT COUNT(*) as count FROM analytics_visits WHERE visited_at >= NOW() - INTERVAL 24 HOUR";
                break;
            case '7d':
                $sql = "SELECT COUNT(*) as count FROM analytics_visits WHERE visited_at >= NOW() - INTERVAL 7 DAY";
                break;
            case '30d':
                $sql = "SELECT COUNT(*) as count FROM analytics_visits WHERE visited_at >= NOW() - INTERVAL 30 DAY";
                break;
        }
        
        return DB::fetch($sql)['count'];
    }
    
    public static function getBrowserStats() {
        return DB::fetchAll("SELECT browser, COUNT(*) as count FROM analytics_visits GROUP BY browser ORDER BY count DESC LIMIT 5");
    }
}
