<?php

require_once __DIR__ . '/Config.php';
require_once __DIR__ . '/Logger.php';

class Indexer {
    
    /**
     * Requests indexing for a URL using Google Indexing API
     * NOTE: Requires a Service Account JSON key file referenced in .env
     */
    public static function requestIndexing($url, $type = 'URL_UPDATED') {
        $keyFile = Config::get('GOOGLE_AUTH_JSON_PATH');
        
        // Resolve relative path if needed
        if ($keyFile && !is_absolute_path($keyFile)) {
            $keyFile = __DIR__ . '/../' . $keyFile;
        }

        if (!$keyFile || !file_exists($keyFile)) {
            Logger::warning("Google Indexing skipped: Credentials file not found at $keyFile");
            return false;
        }

        $auth = json_decode(file_get_contents($keyFile), true);
        if (!$auth) {
            Logger::error("Google Indexing failed: Invalid JSON in credentials file");
            return false;
        }

        try {
            $accessToken = self::getAccessToken($auth);
        } catch (Exception $e) {
            Logger::error("Google Indexing Auth Failed: " . $e->getMessage());
            return false;
        }

        $endpoint = 'https://indexing.googleapis.com/v3/urlNotifications:publish';
        $data = [
            'url' => $url,
            'type' => $type
        ];

        $ch = curl_init($endpoint);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $accessToken
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        Logger::info("Indexing requested for $url. Code: $httpCode. Response: $response");
        
        return $httpCode == 200;
    }

    private static function getAccessToken($authConfig) {
        $now = time();
        $header = json_encode(['alg' => 'RS256', 'typ' => 'JWT']);
        $claim = json_encode([
            'iss' => $authConfig['client_email'],
            'scope' => 'https://www.googleapis.com/auth/indexing',
            'aud' => 'https://oauth2.googleapis.com/token',
            'exp' => $now + 3600,
            'iat' => $now
        ]);

        $base64UrlHeader = self::base64UrlEncode($header);
        $base64UrlClaim = self::base64UrlEncode($claim);
        
        $signatureInput = $base64UrlHeader . "." . $base64UrlClaim;
        
        $signature = '';
        $success = openssl_sign($signatureInput, $signature, $authConfig['private_key'], 'SHA256');
        
        if (!$success) {
            throw new Exception("OpenSSL signing failed");
        }

        $jwt = $signatureInput . "." . self::base64UrlEncode($signature);

        // Exchange JWT for Access Token
        $ch = curl_init('https://oauth2.googleapis.com/token');
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
            'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
            'assertion' => $jwt
        ]));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        $data = json_decode($response, true);
        if (isset($data['access_token'])) {
            return $data['access_token'];
        }
        
        throw new Exception("Failed to get access token: " . $response);
    }

    private static function base64UrlEncode($data) {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }
}

function is_absolute_path($path) {
    return $path === '' || strpos($path, '/') === 0 || preg_match('#^[a-zA-Z]:\\\\#', $path);
}
