<?php

require_once __DIR__ . '/Logger.php';

class Scraper {
    
    public static function getGoogleTrends($geo = 'US') {
        // 1. Generic Daily Trends (Existing)
        $rssUrl = "https://trends.google.com/trending/rss?geo=$geo";
        $trends = self::fetchAndParseRss($rssUrl, 'Google Trends');

        // 2. Specific "Obituary" News Feed (New)
        // This fetches actual news articles about obituaries, which is better for finding new deaths
        // Query: "obituary" AND "died" to be more specific, within last 24h (when:1d)
        $newsRssUrl = "https://news.google.com/rss/search?q=obituary+died+when:1d&hl=en-US&gl=US&ceid=US:en";
        $newsTrends = self::fetchAndParseRss($newsRssUrl, 'Google News');

        // Merge and deduplicate based on title
        $allTrends = array_merge($trends, $newsTrends);
        
        return $allTrends;
    }

    private static function fetchAndParseRss($url, $sourceType) {
        $xml = self::fetchUrl($url);
        
        if (!$xml) {
            Logger::error("Failed to fetch RSS from $sourceType: $url");
            return [];
        }

        $items = [];
        try {
            $rss = new SimpleXMLElement($xml);
            foreach ($rss->channel->item as $item) {
                // Handle different RSS formats
                $newsUrl = (string)$item->link;
                
                // Google Trends specific fields
                if ($sourceType === 'Google Trends') {
                    $newsUrl = (string)$item->children('ht', true)->news_item_url;
                    if (!$newsUrl) $newsUrl = (string)$item->link;
                }

                $items[] = [
                    'title' => (string)$item->title,
                    'traffic' => $sourceType === 'Google Trends' ? (string)$item->children('ht', true)->approx_traffic : 'N/A',
                    'news_item_title' => (string)$item->title,
                    'news_item_url' => $newsUrl,
                    'pubDate' => (string)$item->pubDate,
                    'source_type' => $sourceType
                ];
            }
        } catch (Exception $e) {
            Logger::error("Error parsing XML from $sourceType: " . $e->getMessage());
        }
        
        return $items;
    }

    public static function fetchUrl($url) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 1);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
        
        $data = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            Logger::error("cURL Error fetching $url: $error");
            return null;
        }
        
        return $data;
    }

    public static function extractArticleContent($html) {
        if (!$html) return '';

        $dom = new DOMDocument();
        libxml_use_internal_errors(true);
        $dom->loadHTML($html);
        libxml_clear_errors();

        $xpath = new DOMXPath($dom);
        
        // Remove script and style tags
        foreach ($xpath->query('//script|//style|//nav|//header|//footer|//aside') as $node) {
            $node->parentNode->removeChild($node);
        }

        // Try to find the main content container
        // Common classes/ids for article bodies
        $queries = [
            "//article",
            "//div[contains(@class, 'content')]",
            "//div[contains(@class, 'article')]",
            "//div[contains(@class, 'post-body')]",
            "//main"
        ];

        $content = '';
        foreach ($queries as $query) {
            $nodes = $xpath->query($query);
            if ($nodes->length > 0) {
                foreach ($nodes as $node) {
                    $content .= $dom->saveHTML($node);
                }
                break;
            }
        }

        // Fallback: just get body text if nothing specific found
        if (empty($content)) {
            $body = $dom->getElementsByTagName('body')->item(0);
            if ($body) {
                $content = $dom->saveHTML($body);
            }
        }

        // Strip tags but keep paragraphs
        $content = strip_tags($content, '<p><br><h2><h3><h4><ul><li>');
        
        // Basic cleanup
        $content = preg_replace('/\s+/', ' ', $content);
        $content = trim($content);

        return $content;
    }
}
